;| acmBlockBasispunktAendern

ndert den Blockbasispunkt des Block einer ausgewhlten Blockreferenz und aktualisiert alle Referenzen
des Blocks.
Erscheinungsbild und Lage der Referenzen wird dabei nicht verndert.

Plattform: ab AutoCAD 2020

Copyright
Markus Hoffmann, www.CADmaro.de

Juli 2023
|;
(defun c:acmBlockBasispunktAendern (/ aAD ss e pt)
  (mx:Init)
  (if
    (or
      (setq ss (ssget "_I" '((0 . "INSERT"))))
      (setq ss (ssget "_:S" '((0 . "INSERT"))))
    )
     (if
       (= (vla-get-IsDynamicBlock
            (vlax-ename->vla-object (setq e (ssname ss 0)))
          )
          :vlax-false
       )
        (progn
          (setq pt
                 (getpoint
                   (cdr (assoc 10 (entget e)))
                   "\nNeuer Basispunkt: "
                 )
          )
          (mx:NewBlockBasepoint e pt 'T)
          (princ "\nFertig")
        )
        (alert
          "Der Basispunkt dynamischer Blcke kann mit diesem Tool nicht verndert werden."
        )
     )
  )
  (mx:Reset)
  (princ)
)

 ;|mx:NewBlockBasepoint
Verschiebt den Basispunkt eines Blockes um den angegebenen Vektor

Parameter:
ent [Ename]
lNewPt
flag bei 'T werden die Einfgepunkte der Blockreferenzen angepasst

Beispiel:
(mx:NewBlockBasepoint (car(entsel)) '(100 100 0) 'T)
|;
(defun mx:NewBlockBasepoint (ent lNewPt flag / eInsert vector)
  (setq eInsert (entget ent))
  (setq vector
         (mapcar '- lNewPt (cdr (assoc 10 eInsert)))
  )
  (if
    (not
      (and
        (equal (cdr (assoc 41 eInsert)) 1.0)
        (equal (cdr (assoc 42 eInsert)) 1.0)
        (equal (cdr (assoc 43 eInsert)) 1.0)
        (equal (cdr (assoc 50 eInsert)) 0.0)
      )
    )
     (setq vector
            (mapcar '/
                    (mx:Rotate vector (- 0 (cdr (assoc 50 eInsert))))
                    (list
                      (cdr (assoc 41 eInsert))
                      (cdr (assoc 42 eInsert))
                      (cdr (assoc 43 eInsert))
                    )
            )
     )
  )
  (mx:MoveBlockBasepoint
    (cdr (assoc 2 eInsert))
    vector
    flag
  )
)

 ;|mx:MoveBlockBasepoint
Verschiebt den Basispunkt eines Blockes um den angegebenen Vektor

Parameter:
sName - Blockname
vector - Verschiebevektor
flag bei 'T werden die Einfgepunkte der Blockreferenzen angepasst

Beispiel:
(mx:MoveBlockBasepoint "aa" '(100 100 0) 'T)
|;
(defun mx:MoveBlockBasepoint (sName vector flag / oBlock Origin retval sErr)
  (if
    (and
      (not
        (if
          (vl-catch-all-error-p
            (setq oBlock (vl-catch-all-apply
                           'vla-item
                           (list
                             (vla-get-blocks aAD)
                             sName
                           )
                         )
            )
          )
           (if (not sErr)
             (setq sErr (vl-catch-all-error-message oBlock))
           )
        )
      )
      (or
        (cond
          ((= (length vector) 3) 'T)
          ((= (length vector) 2)
           (setq vector (list (car vector) (cadr vector) 0.0))
          )
        )
        (setq vector '(0.0 0.0 0.0))
      )
      (setq Origin
             (vlax-safearray->list
               (vlax-variant-value
                 (vla-get-Origin oBlock)
               )
             )
      )
      (not
        (vl-catch-all-error-p
          (setq sErr
                 (vl-catch-all-apply
                   'vlax-put-property
                   (list oBlock
                         'Origin
                         (vlax-3d-point '(0.0 0.0 0.0))
                   )
                 )
          )
        )
      )
    )
     (progn
       (vlax-for o oBlock
         (if
           (vl-catch-all-error-p
             (setq
               sErr (vl-catch-all-apply
                      'vlax-invoke-method
                      (list o
                            'Move
                            (vlax-3d-point (mapcar '+ vector Origin))
                            (vlax-3d-point '(0.0 0.0 0.0))
                      )
                    )
             )
           )
            (if (not sErr)
              (setq sErr (vl-catch-all-error-message retval))
            )
         )
       )
       (vlax-for oLayout (vla-get-layouts aAD)
         (vlax-for o
                     (setq oBlock (vla-get-Block oLayout))
           (if
             (and
               (member
                 (strcase (vla-get-objectname o))
                 '("ACDBBLOCKREFERENCE" "ACDBMINSERTBLOCK")
               )
               (= (strcase (vla-get-name o)) (strcase sName))
             )
              (setq
                sErr (mx:MoveInsert o vector flag sErr)
              )
           )
         )
       )
       (vlax-for oBlock (vla-get-blocks aAD)
         (if
           (= (vla-get-islayout oBlock) :vlax-false)
            (vlax-for o oBlock
              (if
                (and (member (strcase (vla-get-objectname o))
                             '("ACDBBLOCKREFERENCE" "ACDBMINSERTBLOCK")
                     )
                     (= (strcase (vla-get-name o)) (strcase sName))
                )
                 (setq sErr (mx:MoveInsert o vector flag sErr))
              )
            )
         )
       )
     )
  )
  (not
    (if sErr
      (setvar "USERS1" sErr)
    )
  )
)

(defun mx:MoveInsert (obj vector flag sErr / retval X Y Z R)
  (setq X (* (car vector) (float (vla-get-XScaleFactor obj))))
  (setq Y (* (cadr vector) (float (vla-get-YScaleFactor obj))))
  (setq Z (* (caddr vector) (float (vla-get-ZScaleFactor obj))))
  (setq R (vla-get-Rotation obj))
  (setq R (+ (cond
               ((and (zerop X) (>= Y 0)) (/ Pi 2.0))
               ((and (zerop X) (< Y 0)) (/ Pi -2.0))
               ((and (< X 0) (>= Y 0)) (+ (atan (/ Y X)) PI))
               ((and (< X 0) (< Y 0)) (- (atan (/ Y X)) PI))
               ('T (atan (/ Y X)))
             )
             R
          )
  )
  (setq vector
         (mapcar
           '(lambda (A)
              (if (< (abs A) 1e-8)
                0.0
                A
              )
            )
           (list
             (* (cos R) (sqrt (+ (* X X) (* Y Y))))
             (* (sin R) (sqrt (+ (* X X) (* Y Y))))
             Z
           )
         )
  )
  (if flag
    (if (vl-catch-all-error-p
          (setq retval
                 (vl-catch-all-apply
                   'vlax-invoke-method
                   (list obj
                         'MOVE
                         (vlax-3d-point '(0 0 0))
                         (vlax-3d-point vector)
                   )
                 )
          )
        )
      (if (not sErr)
        (setq sErr (vl-catch-all-error-message retval))
      )
    )
  )
  (if (= (vla-get-hasattributes obj) :vlax-true)
    (foreach ATTRIBUT (vlax-safearray->list
                        (vlax-variant-value (vla-getattributes obj))
                      )
      (if (vl-catch-all-error-p
            (setq retval (vl-catch-all-apply
                           'vlax-invoke-method
                           (list ATTRIBUT
                                 'MOVE
                                 (vlax-3d-point vector)
                                 (vlax-3d-point '(0 0 0))
                           )
                         )
            )
          )
        (if (not sErr)
          (setq sErr (vl-catch-all-error-message retval))
        )
      )
    )
  )
  sErr
)

(defun mx:Rotate (vector R / X Y Z)
  (setq X (car vector))
  (setq Y (cadr vector))
  (setq Z (caddr vector))
  (setq R (+ (cond
               ((and (zerop X) (>= Y 0)) (/ Pi 2.0))
               ((and (zerop X) (< Y 0)) (/ Pi -2.0))
               ((and (< X 0) (>= Y 0)) (+ (atan (/ Y X)) PI))
               ((and (< X 0) (< Y 0)) (- (atan (/ Y X)) PI))
               ('T (atan (/ Y X)))
             )
             R
          )
  )
  (mapcar '(lambda (A)
             (if (< (abs A) 1e-8)
               0.0
               A
             )
           )
          (list
            (* (cos R) (sqrt (+ (* X X) (* Y Y))))
            (* (sin R) (sqrt (+ (* X X) (* Y Y))))
            Z
          )
  )
)

 ;| mx:Init

Initialisierung
|;
(defun mx:Init ()
  (vl-load-com)
  (setq aAD
         (vla-get-ActiveDocument
           (vlax-get-acad-object)
         )
  )
  (command-s "_ucs" "_w")
  (or command-s (setq command-s command))
  (setq intECHO (getvar "CMDECHO"))
  (setvar "CMDECHO" 0)
  (setq errorMX *error*
        *error* mx:Error
  )
  (vla-EndUndomark aAD)
  (vla-StartUndomark aAD)
)

 ;| mx:Reset

Zurcksetzen
|;
(defun mx:Reset ()
  (command-s "_ucs" "_p")
  (setvar "CMDECHO" intECHO)
  (vla-EndUndomark aAD)
  (vlax-release-object aAD)
  (setq *error* errorMX)
  (mapcar
    '(lambda (arg)
       (set
         arg
         'nil
       )
     )
    (list
      'errorMX
      'intECHO
      'aAD
    )
  )
)

 ;| mx:Error

Errorfunktion
|;
(defun mx:Error (s)
  (print (strcat "Fehler " s))
  (command-s "_.undo" "_back")
  (mx:Reset)
  (princ)
)

;;; Kurzbefehl
(defun c:acmbbae () (c:acmBlockBasispunktAendern))

;; Feedback beim Laden
(princ
  "\nacmBlockBasispunktAendern wurde geladen. Copyright M.Hoffmann, www.CADmaro.de.
Start mit \"acmBlockBasispunktAendern\" oder \"acmbbae\"."
)
(princ)